from django.db.models.signals import post_save, pre_save, m2m_changed
from django.dispatch import receiver
from django.utils import timezone
from .models import Task, Group
from notification.models import Notification


@receiver(pre_save, sender=Task)
def task_pre_save(sender, instance, **kwargs):
    """
    Signal executado antes de salvar uma tarefa.
    """
    # Se a tarefa está sendo criada
    if not instance.pk:
        return
    
    # Verifica se o status mudou
    try:
        old_task = Task.objects.get(pk=instance.pk)
        
        # Se o status mudou para concluída
        if old_task.status != 'concluida' and instance.status == 'concluida':
            instance.tempo_fim = timezone.now()
            
            # Criar notificação de tarefa concluída
            if instance.responsavel:
                Notification.criar_notificacao_tarefa_concluida(instance)
        
        # Se o status mudou de concluída para outro
        elif old_task.status == 'concluida' and instance.status != 'concluida':
            instance.tempo_fim = None
        
        # Se o responsável mudou
        if old_task.responsavel != instance.responsavel and instance.responsavel:
            Notification.criar_notificacao_responsavel_adicionado(instance, instance.responsavel)
            
    except Task.DoesNotExist:
        pass


@receiver(post_save, sender=Task)
def task_post_save(sender, instance, created, **kwargs):
    """
    Signal executado após salvar uma tarefa.
    """
    if created:
        # Tarefa criada - criar notificação se tem responsável
        if instance.responsavel:
            Notification.criar_notificacao_responsavel_adicionado(instance, instance.responsavel)
    
    # Recalcular progresso do grupo, board e projeto
    instance.group.save()  # Isso vai disparar o recálculo em cascata


@receiver(m2m_changed, sender=Task.dependencias.through)
def task_dependencies_changed(sender, instance, action, pk_set, **kwargs):
    """
    Signal executado quando as dependências de uma tarefa mudam.
    """
    if action in ['post_add', 'post_remove', 'post_clear']:
        # Recalcular datas da tarefa baseado nas dependências
        instance.recalcular_datas()


@receiver(post_save, sender=Group)
def group_post_save(sender, instance, **kwargs):
    """
    Signal executado após salvar um grupo.
    """
    # Recalcular progresso do board
    if instance.board:
        instance.board.save()


# Importar signals no apps.py
def ready():
    import task.signals

