from django.test import TestCase, Client as DjangoTestClient
from django.contrib.auth.models import User
from django.urls import reverse
from django.utils import timezone
from datetime import timedelta, date
from decimal import Decimal
import json

from .models import Task, Group
from board.models import Board
from collaborator.models import Collaborator, Area
from project.models import Project
from workspace.models import Workspace
from portfolio.models import Portfolio
from client.models import Client


class TaskModelTest(TestCase):
    """Testes para o modelo Task"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        # Criar usuário
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        # Criar área
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        # Criar colaborador
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva',
            email='joao@example.com',
            area=self.area,
            cargo='Desenvolvedor'
        )
        
        # Criar cliente
        self.cliente = Client.objects.create(
            nome='Cliente Teste LTDA',
            cnpj='12.345.678/0001-90',
            email='cliente@teste.com',
            telefone='(11) 99999-9999',
            endereco='Rua Teste, 123 - São Paulo/SP',
            ativo=True
        )
        
        # Criar workspace
        self.workspace = Workspace.objects.create(
            nome='Workspace Teste',
            descricao='Workspace para testes',
            cliente=self.cliente
        )
        
        # Criar portfolio
        self.portfolio = Portfolio.objects.create(
            nome='Portfolio Teste',
            descricao='Portfolio para testes',
            workspace=self.workspace,
            cliente=self.cliente
        )
        
        # Criar projeto
        self.projeto = Project.objects.create(
            nome='Projeto Teste',
            descricao='Projeto para testes',
            portfolio=self.portfolio,
            workspace=self.workspace,
            cliente=self.cliente,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=30)
        )
        
        # Criar board
        self.board = Board.objects.create(
            nome='Board Teste',
            descricao='Board para testes',
            project=self.projeto,
            workspace=self.workspace,
            cliente=self.cliente
        )
        
        # Criar grupo
        self.grupo = Group.objects.create(
            nome='Desenvolvimento',
            board=self.board,
            tipo='desenvolvimento',
            descricao='Grupo de desenvolvimento'
        )
        
        # Criar tarefa
        self.tarefa = Task.objects.create(
            nome='Tarefa Teste',
            group=self.grupo,
            descricao='Descrição da tarefa teste',
            status='nao_iniciada',
            prioridade='media',
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=5),
            duracao_dias=5,
            esforco_previsto=Decimal('8.0'),
            responsavel=self.colaborador
        )

    def test_criacao_tarefa(self):
        """Testa a criação básica de uma tarefa"""
        self.assertEqual(self.tarefa.nome, 'Tarefa Teste')
        self.assertEqual(self.tarefa.status, 'nao_iniciada')
        self.assertEqual(self.tarefa.prioridade, 'media')
        self.assertEqual(self.tarefa.esforco_previsto, Decimal('8.0'))

    def test_progresso_percentual(self):
        """Testa o cálculo do progresso percentual"""
        # Inicialmente 0%
        self.assertEqual(self.tarefa.progresso_percentual, 0)
        
        # Com esforço utilizado
        self.tarefa.esforco_utilizado = Decimal('4.0')
        self.tarefa.save()
        self.assertEqual(self.tarefa.progresso_percentual, 50.0)

    def test_tarefa_atrasada(self):
        """Testa se a tarefa está atrasada"""
        # Tarefa com data futura não está atrasada
        self.assertFalse(self.tarefa.is_atrasada)
        
        # Tarefa com data passada está atrasada
        from django.utils import timezone
        data_passada = timezone.now().date() - timedelta(days=10)
        self.tarefa.cronograma_fim = data_passada
        # Não usar save() para evitar recálculo automático
        Task.objects.filter(pk=self.tarefa.pk).update(cronograma_fim=data_passada)
        self.tarefa.refresh_from_db()
        

        
        self.assertTrue(self.tarefa.is_atrasada)

    def test_iniciar_trabalho(self):
        """Testa o início do trabalho na tarefa"""
        self.tarefa.iniciar_trabalho()
        self.assertEqual(self.tarefa.status, 'em_andamento')
        self.assertIsNotNone(self.tarefa.tempo_iniciado)
        self.assertIsNone(self.tarefa.tempo_pausado)

    def test_pausar_trabalho(self):
        """Testa a pausa do trabalho na tarefa"""
        self.tarefa.iniciar_trabalho()
        self.tarefa.pausar_trabalho()
        self.assertEqual(self.tarefa.status, 'pausada')
        self.assertIsNotNone(self.tarefa.tempo_pausado)

    def test_concluir_tarefa(self):
        """Testa a conclusão da tarefa"""
        self.tarefa.iniciar_trabalho()
        self.tarefa.concluir_tarefa()
        self.assertEqual(self.tarefa.status, 'concluida')
        self.assertIsNotNone(self.tarefa.data_conclusao)

    def test_dependencias(self):
        """Testa o sistema de dependências entre tarefas"""
        # Criar tarefa dependente
        tarefa_dependente = Task.objects.create(
            nome='Tarefa Dependente',
            group=self.grupo,
            cronograma_inicio=date.today() + timedelta(days=10),
            cronograma_fim=date.today() + timedelta(days=15),
            duracao_dias=5,
            esforco_previsto=Decimal('4.0'),
            responsavel=self.colaborador
        )
        
        # Adicionar dependência
        tarefa_dependente.dependencias.add(self.tarefa)
        
        # Inicialmente não pode iniciar
        self.assertFalse(tarefa_dependente.pode_iniciar)
        
        # Após concluir a dependência, pode iniciar
        self.tarefa.concluir_tarefa()
        tarefa_dependente.refresh_from_db()
        self.assertTrue(tarefa_dependente.pode_iniciar)


class GroupModelTest(TestCase):
    """Testes para o modelo Group"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva',
            email='joao@example.com',
            area=self.area,
            cargo='Desenvolvedor'
        )
        
        # Criar cliente
        self.cliente = Client.objects.create(
            nome='Cliente Teste LTDA',
            cnpj='12.345.678/0001-90',
            email='cliente@teste.com',
            telefone='(11) 99999-9999',
            endereco='Rua Teste, 123 - São Paulo/SP',
            ativo=True
        )
        
        # Criar workspace
        self.workspace = Workspace.objects.create(
            nome='Workspace Teste',
            descricao='Workspace para testes',
            cliente=self.cliente
        )
        
        # Criar portfolio
        self.portfolio = Portfolio.objects.create(
            nome='Portfolio Teste',
            descricao='Portfolio para testes',
            workspace=self.workspace,
            cliente=self.cliente
        )
        
        # Criar projeto
        self.projeto = Project.objects.create(
            nome='Projeto Teste',
            descricao='Projeto para testes',
            portfolio=self.portfolio,
            workspace=self.workspace,
            cliente=self.cliente,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=30)
        )
        
        self.board = Board.objects.create(
            nome='Board Teste',
            descricao='Board para testes',
            project=self.projeto
        )
        
        self.grupo = Group.objects.create(
            nome='Desenvolvimento',
            board=self.board,
            tipo='desenvolvimento',
            descricao='Grupo de desenvolvimento',
            limite_tarefas=5
        )

    def test_criacao_grupo(self):
        """Testa a criação básica de um grupo"""
        self.assertEqual(self.grupo.nome, 'Desenvolvimento')
        self.assertEqual(self.grupo.tipo, 'desenvolvimento')
        self.assertEqual(self.grupo.limite_tarefas, 5)

    def test_total_tasks(self):
        """Testa o cálculo do total de tarefas"""
        self.assertEqual(self.grupo.total_tasks, 0)
        
        # Criar tarefa
        Task.objects.create(
            nome='Tarefa 1',
            group=self.grupo,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=5),
            duracao_dias=5,
            esforco_previsto=Decimal('8.0'),
            responsavel=self.colaborador
        )
        
        self.assertEqual(self.grupo.total_tasks, 1)

    def test_limite_tarefas(self):
        """Testa o limite de tarefas do grupo"""
        # Criar tarefas até o limite
        for i in range(5):
            Task.objects.create(
                nome=f'Tarefa {i+1}',
                group=self.grupo,
                cronograma_inicio=date.today(),
                cronograma_fim=date.today() + timedelta(days=5),
                duracao_dias=5,
                esforco_previsto=Decimal('8.0'),
                responsavel=self.colaborador
            )
        
        self.assertTrue(self.grupo.is_limite_atingido)
        self.assertFalse(self.grupo.pode_receber_tarefa())

    def test_progresso_percentual(self):
        """Testa o cálculo do progresso percentual do grupo"""
        # Criar tarefas com diferentes status
        Task.objects.create(
            nome='Tarefa Concluída',
            group=self.grupo,
            status='concluida',
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=5),
            duracao_dias=5,
            esforco_previsto=Decimal('8.0'),
            responsavel=self.colaborador
        )
        
        Task.objects.create(
            nome='Tarefa Em Andamento',
            group=self.grupo,
            status='em_andamento',
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=5),
            duracao_dias=5,
            esforco_previsto=Decimal('8.0'),
            responsavel=self.colaborador
        )
        
        self.assertEqual(self.grupo.progresso_percentual, 50.0)


class TaskViewsTest(TestCase):
    """Testes para as views do app task"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_instance = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva',
            email='joao@example.com',
            area=self.area,
            cargo='Desenvolvedor'
        )
        
        # Criar cliente
        self.cliente = Client.objects.create(
            nome='Cliente Teste LTDA',
            cnpj='12.345.678/0001-90',
            email='cliente@teste.com',
            telefone='(11) 99999-9999',
            endereco='Rua Teste, 123 - São Paulo/SP',
            ativo=True
        )
        
        # Criar workspace
        self.workspace = Workspace.objects.create(
            nome='Workspace Teste',
            descricao='Workspace para testes',
            cliente=self.cliente
        )
        
        # Criar portfolio
        self.portfolio = Portfolio.objects.create(
            nome='Portfolio Teste',
            descricao='Portfolio para testes',
            workspace=self.workspace,
            cliente=self.cliente
        )
        
        # Criar projeto
        self.projeto = Project.objects.create(
            nome='Projeto Teste',
            descricao='Projeto para testes',
            portfolio=self.portfolio,
            workspace=self.workspace,
            cliente=self.cliente,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=30)
        )
        
        self.board = Board.objects.create(
            nome='Board Teste',
            descricao='Board para testes',
            project=self.projeto
        )
        
        self.grupo = Group.objects.create(
            nome='Desenvolvimento',
            board=self.board,
            tipo='desenvolvimento',
            descricao='Grupo de desenvolvimento'
        )
        
        self.tarefa = Task.objects.create(
            nome='Tarefa Teste',
            group=self.grupo,
            descricao='Descrição da tarefa teste',
            status='nao_iniciada',
            prioridade='media',
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=5),
            duracao_dias=5,
            esforco_previsto=Decimal('8.0'),
            responsavel=self.colaborador
        )

    def test_lista_tarefas_sem_login(self):
        """Testa acesso à lista de tarefas sem login"""
        response = self.client_instance.get(reverse('task:list'))
        self.assertEqual(response.status_code, 302)  # Redirecionamento para login

    def test_lista_tarefas_com_login(self):
        """Testa acesso à lista de tarefas com login"""
        self.client_instance.login(username='testuser', password='testpass123')
        response = self.client_instance.get(reverse('task:list'))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Tarefa Teste')

    def test_detalhe_tarefa(self):
        """Testa a view de detalhe da tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        response = self.client_instance.get(reverse('task:detail', kwargs={'pk': self.tarefa.pk}))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Tarefa Teste')

    def test_criar_tarefa(self):
        """Testa a criação de uma nova tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        
        data = {
            'nome': 'Nova Tarefa',
            'group': self.grupo.pk,
            'descricao': 'Descrição da nova tarefa',
            'status': 'nao_iniciada',
            'prioridade': 'alta',
            'cronograma_inicio': date.today(),
            'cronograma_fim': date.today() + timedelta(days=3),
            'duracao_dias': 3,
            'esforco_previsto': '6.0',
            'responsavel': self.colaborador.pk
        }
        
        response = self.client_instance.post(reverse('task:create'), data)
        self.assertEqual(response.status_code, 302)  # Redirecionamento após criação
        
        # Verificar se a tarefa foi criada
        nova_tarefa = Task.objects.get(nome='Nova Tarefa')
        self.assertEqual(nova_tarefa.prioridade, 'alta')

    def test_editar_tarefa(self):
        """Testa a edição de uma tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        
        data = {
            'nome': 'Tarefa Editada',
            'group': self.grupo.pk,
            'descricao': 'Descrição editada',
            'status': 'em_andamento',
            'prioridade': 'alta',
            'cronograma_inicio': date.today(),
            'cronograma_fim': date.today() + timedelta(days=5),
            'duracao_dias': 5,
            'esforco_previsto': '10.0',
            'responsavel': self.colaborador.pk
        }
        
        response = self.client_instance.post(
            reverse('task:edit', kwargs={'pk': self.tarefa.pk}),
            data
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se a tarefa foi editada
        self.tarefa.refresh_from_db()
        self.assertEqual(self.tarefa.nome, 'Tarefa Editada')
        self.assertEqual(self.tarefa.status, 'em_andamento')

    def test_deletar_tarefa(self):
        """Testa a exclusão de uma tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        
        response = self.client_instance.post(
            reverse('task:delete', kwargs={'pk': self.tarefa.pk})
        )
        self.assertEqual(response.status_code, 302)
        
        # Verificar se a tarefa foi deletada
        with self.assertRaises(Task.DoesNotExist):
            Task.objects.get(pk=self.tarefa.pk)

    def test_minhas_tarefas(self):
        """Testa a view de minhas tarefas"""
        self.client_instance.login(username='testuser', password='testpass123')
        response = self.client_instance.get(reverse('task:my_tasks'))
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'Tarefa Teste')

    def test_modal_criar_tarefa(self):
        """Testa o modal de criação de tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        response = self.client_instance.get(reverse('task:create_modal'))
        self.assertEqual(response.status_code, 200)

    def test_modal_editar_tarefa(self):
        """Testa o modal de edição de tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        response = self.client_instance.get(
            reverse('task:edit_modal', kwargs={'pk': self.tarefa.pk})
        )
        self.assertEqual(response.status_code, 200)


class TaskAPITest(TestCase):
    """Testes para a API do app task"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_instance = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva',
            email='joao@example.com',
            area=self.area,
            cargo='Desenvolvedor'
        )
        
        # Criar cliente
        self.cliente = Client.objects.create(
            nome='Cliente Teste LTDA',
            cnpj='12.345.678/0001-90',
            email='cliente@teste.com',
            telefone='(11) 99999-9999',
            endereco='Rua Teste, 123 - São Paulo/SP',
            ativo=True
        )
        
        # Criar workspace
        self.workspace = Workspace.objects.create(
            nome='Workspace Teste',
            descricao='Workspace para testes',
            cliente=self.cliente
        )
        
        # Criar portfolio
        self.portfolio = Portfolio.objects.create(
            nome='Portfolio Teste',
            descricao='Portfolio para testes',
            workspace=self.workspace,
            cliente=self.cliente
        )
        
        # Criar projeto
        self.projeto = Project.objects.create(
            nome='Projeto Teste',
            descricao='Projeto para testes',
            portfolio=self.portfolio,
            workspace=self.workspace,
            cliente=self.cliente,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=30)
        )
        
        self.board = Board.objects.create(
            nome='Board Teste',
            descricao='Board para testes',
            project=self.projeto
        )
        
        self.grupo = Group.objects.create(
            nome='Desenvolvimento',
            board=self.board,
            tipo='desenvolvimento',
            descricao='Grupo de desenvolvimento'
        )
        
        self.tarefa = Task.objects.create(
            nome='Tarefa Teste',
            group=self.grupo,
            descricao='Descrição da tarefa teste',
            status='nao_iniciada',
            prioridade='media',
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=5),
            duracao_dias=5,
            esforco_previsto=Decimal('8.0'),
            responsavel=self.colaborador
        )

    def test_api_lista_tarefas(self):
        """Testa a API de listagem de tarefas"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get('/api/tasks/')
        self.assertEqual(response.status_code, 200)
        
        data = json.loads(response.content)
        # Verificar se a tarefa de teste está na lista
        tarefa_encontrada = any(isinstance(item, dict) and item.get('nome') == 'Tarefa Teste' for item in data)
        self.assertTrue(tarefa_encontrada, "Tarefa de teste não encontrada na lista")

    def test_api_detalhe_tarefa(self):
        """Testa a API de detalhe da tarefa"""
        self.client_instance.force_login(self.user)
        response = self.client_instance.get(f'/api/tasks/{self.tarefa.pk}/')
        self.assertEqual(response.status_code, 200)
        
        data = json.loads(response.content)
        self.assertEqual(data['nome'], 'Tarefa Teste')

    def test_api_criar_tarefa(self):
        """Testa a API de criação de tarefa"""
        self.client_instance.force_login(self.user)
        
        data = {
            'nome': 'Nova Tarefa API',
            'group': self.grupo.pk,
            'descricao': 'Descrição da nova tarefa via API',
            'status': 'nao_iniciada',
            'prioridade': 'alta',
            'cronograma_inicio': date.today().isoformat(),
            'cronograma_fim': (date.today() + timedelta(days=3)).isoformat(),
            'duracao_dias': 3,
            'esforco_previsto': '6.0',
            'responsavel': self.colaborador.pk
        }
        
        response = self.client_instance.post(
            '/api/tasks/',
            data=json.dumps(data),
            content_type='application/json'
        )
        self.assertEqual(response.status_code, 201)
        
        # Verificar se a tarefa foi criada
        nova_tarefa = Task.objects.get(nome='Nova Tarefa API')
        self.assertEqual(nova_tarefa.prioridade, 'alta')

    def test_api_atualizar_tarefa(self):
        """Testa a API de atualização de tarefa"""
        self.client_instance.force_login(self.user)
        
        data = {
            'nome': 'Tarefa Atualizada API',
            'group': self.grupo.pk,
            'descricao': 'Descrição atualizada via API',
            'status': 'em_andamento',
            'prioridade': 'alta',
            'cronograma_inicio': date.today().isoformat(),
            'cronograma_fim': (date.today() + timedelta(days=5)).isoformat(),
            'duracao_dias': 5,
            'esforco_previsto': '10.0',
            'responsavel': self.colaborador.pk
        }
        
        response = self.client_instance.put(
            f'/api/tasks/{self.tarefa.pk}/',
            data=json.dumps(data),
            content_type='application/json'
        )
        self.assertEqual(response.status_code, 200)
        
        # Verificar se a tarefa foi atualizada
        self.tarefa.refresh_from_db()
        self.assertEqual(self.tarefa.nome, 'Tarefa Atualizada API')

    def test_api_deletar_tarefa(self):
        """Testa a API de exclusão de tarefa"""
        self.client_instance.force_login(self.user)
        
        response = self.client_instance.delete(f'/api/tasks/{self.tarefa.pk}/')
        self.assertEqual(response.status_code, 204)
        
        # Verificar se a tarefa foi deletada
        with self.assertRaises(Task.DoesNotExist):
            Task.objects.get(pk=self.tarefa.pk)


class TaskIntegrationTest(TestCase):
    """Testes de integração para o app task"""
    
    def setUp(self):
        """Configuração inicial para os testes"""
        self.client_instance = DjangoTestClient()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123',
            email='test@example.com'
        )
        
        self.area = Area.objects.create(
            nome='Desenvolvimento',
            descricao='Área de desenvolvimento'
        )
        
        self.colaborador = Collaborator.objects.create(
            user=self.user,
            nome='João Silva',
            email='joao@example.com',
            area=self.area,
            cargo='Desenvolvedor'
        )
        
        # Criar cliente
        self.cliente = Client.objects.create(
            nome='Cliente Teste LTDA',
            cnpj='12.345.678/0001-90',
            email='cliente@teste.com',
            telefone='(11) 99999-9999',
            endereco='Rua Teste, 123 - São Paulo/SP',
            ativo=True
        )
        
        # Criar workspace
        self.workspace = Workspace.objects.create(
            nome='Workspace Teste',
            descricao='Workspace para testes',
            cliente=self.cliente
        )
        
        # Criar portfolio
        self.portfolio = Portfolio.objects.create(
            nome='Portfolio Teste',
            descricao='Portfolio para testes',
            workspace=self.workspace,
            cliente=self.cliente
        )
        
        # Criar projeto
        self.projeto = Project.objects.create(
            nome='Projeto Teste',
            descricao='Projeto para testes',
            portfolio=self.portfolio,
            workspace=self.workspace,
            cliente=self.cliente,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=30)
        )
        
        self.board = Board.objects.create(
            nome='Board Teste',
            descricao='Board para testes',
            project=self.projeto
        )
        
        self.grupo = Group.objects.create(
            nome='Desenvolvimento',
            board=self.board,
            tipo='desenvolvimento',
            descricao='Grupo de desenvolvimento'
        )

    def test_fluxo_completo_tarefa(self):
        """Testa o fluxo completo de uma tarefa"""
        self.client_instance.login(username='testuser', password='testpass123')
        
        # 1. Criar tarefa
        data = {
            'nome': 'Tarefa Completa',
            'group': self.grupo.pk,
            'descricao': 'Teste de fluxo completo',
            'status': 'nao_iniciada',
            'prioridade': 'alta',
            'cronograma_inicio': date.today(),
            'cronograma_fim': date.today() + timedelta(days=5),
            'duracao_dias': 5,
            'esforco_previsto': '8.0',
            'responsavel': self.colaborador.pk
        }
        
        response = self.client_instance.post(reverse('task:create'), data)
        self.assertEqual(response.status_code, 302)
        
        tarefa = Task.objects.get(nome='Tarefa Completa')
        self.assertEqual(tarefa.status, 'nao_iniciada')
        
        # 2. Iniciar trabalho
        tarefa.iniciar_trabalho()
        self.assertEqual(tarefa.status, 'em_andamento')
        
        # 3. Pausar trabalho
        tarefa.pausar_trabalho()
        self.assertEqual(tarefa.status, 'pausada')
        
        # 4. Retomar trabalho
        tarefa.iniciar_trabalho()
        self.assertEqual(tarefa.status, 'em_andamento')
        
        # Atualizar a referência da tarefa
        tarefa.refresh_from_db()
        
        # 5. Concluir tarefa
        tarefa.concluir_tarefa()
        self.assertEqual(tarefa.status, 'concluida')
        self.assertIsNotNone(tarefa.data_conclusao)

    def test_estatisticas_grupo(self):
        """Testa as estatísticas do grupo"""
        # Criar tarefas com diferentes status
        status_list = ['nao_iniciada', 'em_andamento', 'pausada', 'concluida']
        
        for i, status in enumerate(status_list):
            Task.objects.create(
                nome=f'Tarefa {status}',
                group=self.grupo,
                status=status,
                cronograma_inicio=date.today(),
                cronograma_fim=date.today() + timedelta(days=5),
                duracao_dias=5,
                esforco_previsto=Decimal('8.0'),
                responsavel=self.colaborador
            )
        
        # Verificar estatísticas
        self.assertEqual(self.grupo.total_tasks, 4)
        self.assertEqual(self.grupo.tasks_concluidas, 1)
        self.assertEqual(self.grupo.tasks_em_andamento, 1)
        self.assertEqual(self.grupo.tasks_pausadas, 1)
        self.assertEqual(self.grupo.tasks_nao_iniciadas, 1)
        self.assertEqual(self.grupo.progresso_percentual, 25.0)

    def test_dependencias_cascata(self):
        """Testa o sistema de dependências em cascata"""
        # Criar tarefa base
        tarefa_base = Task.objects.create(
            nome='Tarefa Base',
            group=self.grupo,
            cronograma_inicio=date.today(),
            cronograma_fim=date.today() + timedelta(days=3),
            duracao_dias=3,
            esforco_previsto=Decimal('6.0'),
            responsavel=self.colaborador
        )
        
        # Criar tarefa dependente
        tarefa_dependente = Task.objects.create(
            nome='Tarefa Dependente',
            group=self.grupo,
            cronograma_inicio=date.today() + timedelta(days=5),
            cronograma_fim=date.today() + timedelta(days=8),
            duracao_dias=3,
            esforco_previsto=Decimal('4.0'),
            responsavel=self.colaborador
        )
        
        # Adicionar dependência
        tarefa_dependente.dependencias.add(tarefa_base)
        
        # Verificar que não pode iniciar
        self.assertFalse(tarefa_dependente.pode_iniciar)
        
        # Concluir tarefa base
        tarefa_base.concluir_tarefa()
        tarefa_dependente.refresh_from_db()
        
        # Verificar que agora pode iniciar
        self.assertTrue(tarefa_dependente.pode_iniciar)
