from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q, Count
from django.http import JsonResponse
from django.views.generic import CreateView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from django.utils import timezone
from .models import Task, Group
from .forms import TaskForm, TaskCreateForm, GroupForm
from board.models import Board
from project.models import Project
from collaborator.models import Collaborator
from system_logs.utils import log_object_created, log_object_updated, log_object_deleted, log_system_action


@login_required
def task_list(request):
    """
    Lista todas as tarefas com filtros e paginação.
    """
    # Obter parâmetros de filtro
    search = request.GET.get('search', '')
    status = request.GET.get('status', '')
    board_id = request.GET.get('board', '')
    project_id = request.GET.get('project', '')
    responsavel_id = request.GET.get('responsavel', '')
    
    # Query base
    tasks = Task.objects.select_related('group', 'group__board', 'responsavel').all()
    
    # Aplicar filtros
    if search:
        tasks = tasks.filter(
            Q(nome__icontains=search) |
            Q(group__nome__icontains=search) |
            Q(group__board__nome__icontains=search) |
            Q(descricao__icontains=search)
        )
    
    if status:
        tasks = tasks.filter(status=status)
    
    if board_id:
        tasks = tasks.filter(group__board_id=board_id)
    
    if project_id:
        tasks = tasks.filter(group__board__project_id=project_id)
    
    if responsavel_id:
        tasks = tasks.filter(responsavel_id=responsavel_id)
    
    # Ordenar por grupo e data de início
    tasks = tasks.order_by('group__board__nome', 'group__nome', 'cronograma_inicio')
    
    # Paginação
    paginator = Paginator(tasks, 15)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas
    total_tasks = Task.objects.count()
    tasks_concluidas = Task.objects.filter(status='concluida').count()
    tasks_em_andamento = Task.objects.filter(status='em_andamento').count()
    tasks_nao_iniciadas = Task.objects.filter(status='nao_iniciada').count()
    
    # Lista de boards, projetos e colaboradores para o filtro
    boards = Board.objects.all().order_by('project__nome', 'nome')
    projects = Project.objects.all().order_by('portfolio__nome', 'nome')
    colaboradores = Collaborator.objects.filter(ativo=True).order_by('nome')
    
    context = {
        'tasks': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'search': search,
        'status': status,
        'board_id': board_id,
        'project_id': project_id,
        'responsavel_id': responsavel_id,
        'total_tasks': total_tasks,
        'tasks_concluidas': tasks_concluidas,
        'tasks_em_andamento': tasks_em_andamento,
        'tasks_nao_iniciadas': tasks_nao_iniciadas,
        'boards': boards,
        'projects': projects,
        'colaboradores': colaboradores,
    }
    
    return render(request, 'task/list.html', context)


@login_required
def task_create(request):
    """
    Cria uma nova tarefa.
    """
    if request.method == 'POST':
        form = TaskForm(request.POST)
        if form.is_valid():
            task = form.save()
            
            # Adicionar dependências
            dependencias_ids = request.POST.getlist('dependencias')
            if dependencias_ids:
                dependencias = Task.objects.filter(id__in=dependencias_ids)
                task.dependencias.set(dependencias)
            
            # Registrar log
            log_object_created(request, task)
            
            messages.success(request, f'Tarefa "{task.nome}" criada com sucesso!')
            return redirect('task:detail', pk=task.pk)
        else:
            messages.error(request, 'Por favor, corrija os erros no formulário.')
    else:
        form = TaskForm()
    
    # GET request - mostrar formulário
    groups = Group.objects.filter(ativo=True).order_by('board__nome', 'nome')
    colaboradores = Collaborator.objects.filter(ativo=True).order_by('nome')
    todas_tarefas = Task.objects.select_related('group', 'group__board', 'group__board__project').all().order_by('nome')
    
    context = {
        'form': form,
        'groups': groups,
        'colaboradores': colaboradores,
        'todas_tarefas': todas_tarefas,
        'task': None,
    }
    
    return render(request, 'task/form.html', context)


@login_required
def task_detail(request, pk):
    """
    Exibe os detalhes de uma tarefa.
    """
    task = get_object_or_404(Task, pk=pk)
    
    context = {
        'task': task,
    }
    
    return render(request, 'task/detail.html', context)


@login_required
def task_edit(request, pk):
    """
    Edita uma tarefa existente.
    """
    task = get_object_or_404(Task, pk=pk)
    
    if request.method == 'POST':
        form = TaskForm(request.POST, instance=task)
        if form.is_valid():
            task = form.save()
            
            # Atualizar dependências
            dependencias_ids = request.POST.getlist('dependencias')
            if dependencias_ids:
                dependencias = Task.objects.filter(id__in=dependencias_ids)
                task.dependencias.set(dependencias)
            else:
                task.dependencias.clear()
            
            # Registrar log
            log_object_updated(request, task)
            
            messages.success(request, f'Tarefa "{task.nome}" atualizada com sucesso!')
            return redirect('task:detail', pk=task.pk)
        else:
            messages.error(request, 'Por favor, corrija os erros no formulário.')
    else:
        form = TaskForm(instance=task)
    
    # GET request - mostrar formulário
    groups = Group.objects.filter(ativo=True).order_by('board__nome', 'nome')
    colaboradores = Collaborator.objects.filter(ativo=True).order_by('nome')
    todas_tarefas = Task.objects.select_related('group', 'group__board', 'group__board__project').all().order_by('nome')
    
    context = {
        'task': task,
        'form': form,
        'groups': groups,
        'colaboradores': colaboradores,
        'todas_tarefas': todas_tarefas,
    }
    
    return render(request, 'task/form.html', context)


@login_required
def task_delete(request, pk):
    """
    Exclui uma tarefa.
    """
    task = get_object_or_404(Task, pk=pk)
    
    if request.method == 'POST':
        nome = task.nome
        # Registrar log antes de deletar
        log_object_deleted(request, task)
        task.delete()
        messages.success(request, f'Tarefa "{nome}" excluída com sucesso!')
        return redirect('task:list')
    
    context = {
        'task': task,
    }
    
    return render(request, 'task/delete.html', context)


@login_required
def my_tasks(request):
    """
    Lista as tarefas do usuário logado.
    """
    # Obter parâmetros de filtro
    status = request.GET.get('status', '')
    prioridade = request.GET.get('prioridade', '')
    atrasadas = request.GET.get('atrasadas', '')
    
    # Query base - tarefas do usuário logado
    tasks = Task.objects.filter(responsavel__user=request.user).select_related(
        'group', 'group__board', 'responsavel'
    )
    
    # Aplicar filtros
    if status:
        tasks = tasks.filter(status=status)
    
    if prioridade:
        tasks = tasks.filter(prioridade=prioridade)
    
    if atrasadas == 'sim':
        tasks = tasks.filter(cronograma_fim__lt=timezone.now().date(), status__in=['nao_iniciada', 'em_andamento', 'pausada'])
    elif atrasadas == 'nao':
        tasks = tasks.filter(
            Q(cronograma_fim__gte=timezone.now().date()) | 
            Q(status='concluida') | 
            Q(status='cancelada')
        )
    
    # Ordenar por prioridade e data
    tasks = tasks.order_by('-prioridade', 'cronograma_inicio')
    
    # Paginação
    paginator = Paginator(tasks, 15)
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas
    total_my_tasks = Task.objects.filter(responsavel__user=request.user).count()
    my_tasks_concluidas = Task.objects.filter(responsavel__user=request.user, status='concluida').count()
    my_tasks_em_andamento = Task.objects.filter(responsavel__user=request.user, status='em_andamento').count()
    my_tasks_pendentes = Task.objects.filter(responsavel__user=request.user, status='nao_iniciada').count()
    
    context = {
        'tasks': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'status': status,
        'prioridade': prioridade,
        'atrasadas': atrasadas,
        'total_my_tasks': total_my_tasks,
        'my_tasks_concluidas': my_tasks_concluidas,
        'my_tasks_em_andamento': my_tasks_em_andamento,
        'my_tasks_pendentes': my_tasks_pendentes,
    }
    
    return render(request, 'task/my_tasks.html', context)


@login_required
def iniciar_tarefa(request, pk):
    """
    Inicia uma tarefa.
    """
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk)
        
        # Verificar se o usuário é o responsável
        if not task.responsavel or task.responsavel.user != request.user:
            return JsonResponse({
                'success': False,
                'message': 'Você não é o responsável por esta tarefa'
            })
        
        try:
            task.iniciar_trabalho()
            return JsonResponse({
                'success': True,
                'message': 'Tarefa iniciada com sucesso!',
                'status': task.get_status_display(),
                'esta_trabalhando': task.esta_trabalhando,
                'tempo_total': task.tempo_total_formatado,
                'tempo_hoje': task.tempo_hoje_formatado
            })
        except ValueError as e:
            return JsonResponse({
                'success': False,
                'message': str(e)
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': f'Erro inesperado: {str(e)}'
            })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def pausar_tarefa(request, pk):
    """
    Pausa uma tarefa.
    """
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk)
        
        # Verificar se o usuário é o responsável
        if not task.responsavel or task.responsavel.user != request.user:
            return JsonResponse({
                'success': False,
                'message': 'Você não é o responsável por esta tarefa'
            })
        
        try:
            task.pausar_trabalho()
            return JsonResponse({
                'success': True,
                'message': 'Tarefa pausada com sucesso!',
                'status': task.get_status_display(),
                'esta_trabalhando': task.esta_trabalhando,
                'tempo_total': task.tempo_total_formatado,
                'tempo_hoje': task.tempo_hoje_formatado,
                'esforco_utilizado': float(task.esforco_utilizado)
            })
        except ValueError as e:
            return JsonResponse({
                'success': False,
                'message': str(e)
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': f'Erro inesperado: {str(e)}'
            })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def concluir_tarefa(request, pk):
    """
    Conclui uma tarefa.
    """
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk, responsavel__user=request.user)
        try:
            task.concluir_tarefa()
            return JsonResponse({
                'success': True,
                'message': 'Tarefa concluída com sucesso!',
                'status': task.get_status_display(),
                'tempo_total': task.tempo_total_formatado,
                'esforco_utilizado': float(task.esforco_utilizado),
                'progresso_percentual': float(task.progresso_percentual)
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': str(e)
            })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def retomar_tarefa(request, pk):
    """
    Retoma uma tarefa pausada.
    """
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk, responsavel__user=request.user)
        try:
            task.retomar_trabalho()
            return JsonResponse({
                'success': True,
                'message': 'Tarefa retomada com sucesso!',
                'status': task.get_status_display(),
                'tempo_total': task.tempo_total_formatado,
                'tempo_hoje': task.tempo_hoje_formatado
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': str(e)
            })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})








@login_required
def obter_tempo_tarefa(request, pk):
    """
    Retorna informações de tempo de uma tarefa em tempo real.
    """
    if request.method == 'GET':
        task = get_object_or_404(Task, pk=pk, responsavel__user=request.user)
        return JsonResponse({
            'success': True,
            'tempo_total': task.tempo_total_formatado,
            'tempo_hoje': task.tempo_hoje_formatado,
            'tempo_sessao': task.tempo_sessao_formatado,
            'esta_trabalhando': task.esta_trabalhando,
            'status': task.get_status_display(),
            'esforco_utilizado': float(task.esforco_utilizado),
            'progresso_percentual': float(task.progresso_percentual)
        })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def debug_tarefa(request, pk):
    """
    View de debug para verificar o status de uma tarefa.
    """
    task = get_object_or_404(Task, pk=pk)
    
    debug_info = {
        'id': task.id,
        'nome': task.nome,
        'status': task.status,
        'status_display': task.get_status_display(),
        'tempo_iniciado': task.tempo_iniciado,
        'tempo_pausado': task.tempo_pausado,
        'tempo_total_trabalhado': str(task.tempo_total_trabalhado),
        'esforco_utilizado': float(task.esforco_utilizado),
        'esta_trabalhando': task.esta_trabalhando,
        'tempo_atual_sessao': str(task.tempo_atual_sessao),
        'tempo_sessao_formatado': task.tempo_sessao_formatado,
        'tempo_total_formatado': task.tempo_total_formatado,
        'tempo_hoje_formatado': task.tempo_hoje_formatado,
        'pode_iniciar': task.pode_iniciar,
        'responsavel': task.responsavel.nome if task.responsavel else None,
        'user_atual': request.user.username,
        'user_is_responsavel': task.responsavel and task.responsavel.user == request.user,
    }
    
    return JsonResponse(debug_info)


@login_required
def reset_tarefa(request, pk):
    """
    Reseta o estado da tarefa para corrigir inconsistências.
    """
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk)
        
        # Verificar se o usuário é o responsável
        if not task.responsavel or task.responsavel.user != request.user:
            return JsonResponse({
                'success': False,
                'message': 'Você não é o responsável por esta tarefa'
            })
        
        try:
            # Resetar campos de tempo
            task.tempo_iniciado = None
            task.tempo_pausado = None
            
            # Manter o status atual mas garantir que não está trabalhando
            if task.status == 'em_andamento':
                task.status = 'nao_iniciada'
            
            task.save()
            
            return JsonResponse({
                'success': True,
                'message': 'Estado da tarefa resetado com sucesso!',
                'status': task.get_status_display(),
                'esta_trabalhando': task.esta_trabalhando,
                'tempo_total': task.tempo_total_formatado,
                'tempo_hoje': task.tempo_hoje_formatado,
                'esforco_utilizado': float(task.esforco_utilizado)
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': f'Erro ao resetar tarefa: {str(e)}'
            })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def obter_logs_sessoes(request, pk):
    """
    Retorna os logs de sessões de uma tarefa.
    """
    if request.method == 'GET':
        task = get_object_or_404(Task, pk=pk, responsavel__user=request.user)
        data = request.GET.get('data')
        
        logs = task.obter_logs_por_data(data) if data else task.logs_sessoes
        
        return JsonResponse({
            'success': True,
            'logs': logs,
            'tempo_total': task.tempo_total_formatado,
            'tempo_hoje': task.tempo_hoje_formatado
        })
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def editar_log_sessao(request, pk):
    """
    Edita um log de sessão específico.
    """
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk, responsavel__user=request.user)
        
        try:
            log_id = int(request.POST.get('log_id'))
            nova_duracao = int(request.POST.get('duracao_segundos'))
            nova_data = request.POST.get('data')
            
            if task.editar_log_sessao(log_id, nova_duracao, nova_data):
                task.save()
                
                return JsonResponse({
                    'success': True,
                    'message': 'Log de sessão atualizado com sucesso!',
                    'tempo_total': task.tempo_total_formatado,
                    'tempo_hoje': task.tempo_hoje_formatado,
                    'esforco_utilizado': float(task.esforco_utilizado)
                })
            else:
                return JsonResponse({
                    'success': False,
                    'message': 'Log de sessão não encontrado'
                })
                
        except (ValueError, TypeError) as e:
            return JsonResponse({
                'success': False,
                'message': f'Dados inválidos: {str(e)}'
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': f'Erro ao editar log: {str(e)}'
            })
    
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


@login_required
def deletar_log_sessao(request, pk):
    """
    Deleta um log de sessão específico.
    """
    from datetime import timedelta
    
    if request.method == 'POST':
        task = get_object_or_404(Task, pk=pk, responsavel__user=request.user)
        
        try:
            log_id = int(request.POST.get('log_id'))
            
            # Encontrar e remover o log
            for i, log in enumerate(task.logs_sessoes):
                if log['id'] == log_id:
                    # Calcular diferença para subtrair do tempo total
                    duracao = timedelta(seconds=log['duracao_segundos'])
                    task.tempo_total_trabalhado -= duracao
                    
                    # Atualizar esforço utilizado
                    horas = duracao.total_seconds() / 3600
                    from decimal import Decimal
                    task.esforco_utilizado -= Decimal(str(horas))
                    
                    # Remover o log
                    task.logs_sessoes.pop(i)
                    task.save()
                    
                    return JsonResponse({
                        'success': True,
                        'message': 'Log de sessão removido com sucesso!',
                        'tempo_total': task.tempo_total_formatado,
                        'tempo_hoje': task.tempo_hoje_formatado,
                        'esforco_utilizado': float(task.esforco_utilizado)
                    })
            
            return JsonResponse({
                'success': False,
                'message': 'Log de sessão não encontrado'
            })
                
        except (ValueError, TypeError) as e:
            return JsonResponse({
                'success': False,
                'message': f'Dados inválidos: {str(e)}'
            })
        except Exception as e:
            return JsonResponse({
                'success': False,
                'message': f'Erro ao deletar log: {str(e)}'
            })
    
    return JsonResponse({'success': False, 'message': 'Método não permitido'})


class TaskCreateModalView(LoginRequiredMixin, CreateView):
    """
    View para criação de task via modal.
    """
    model = Task
    form_class = TaskCreateForm
    template_name = 'task/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Filtrar grupos apenas do projeto atual se especificado
        project_id = self.request.GET.get('project')
        if project_id:
            try:
                from project.models import Project
                project = Project.objects.get(id=project_id)
                # Filtrar grupos apenas dos boards do projeto
                context['form'].fields['group'].queryset = Group.objects.filter(
                    board__project=project,
                    ativo=True
                ).order_by('board__nome', 'nome')
            except Project.DoesNotExist:
                # Se o projeto não existir, mostrar todos os grupos ativos
                context['form'].fields['group'].queryset = Group.objects.filter(
                    ativo=True
                ).order_by('board__nome', 'nome')
        else:
            # Se não há projeto especificado, mostrar todos os grupos ativos
            context['form'].fields['group'].queryset = Group.objects.filter(
                ativo=True
            ).order_by('board__nome', 'nome')
        
        context.update({
            'modal_id': 'createTaskModal',
            'modal_title': 'Nova Tarefa',
            'modal_icon': 'bi-list-task',
            'form_id': 'taskForm',
            'submit_text': 'Criar Tarefa'
        })
        return context
    
    def form_valid(self, form):
        try:
            task = form.save(commit=False)
            # Definir valores padrão para campos obrigatórios
            task.duracao_dias = 1
            task.esforco_previsto = 0
            task.esforco_utilizado = 0
            task.save()
            
            # Registrar log
            log_object_created(self.request, task)
            
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': True,
                    'message': 'Tarefa criada com sucesso!',
                    'obj': {
                        'id': task.id,
                        'nome': task.nome,
                        'detail_url': reverse_lazy('task:detail', kwargs={'pk': task.id})
                    }
                })
            return super().form_valid(form)
        except Exception as e:
            print(f"Erro ao criar tarefa: {e}")
            if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
                return JsonResponse({
                    'success': False,
                    'errors': {'__all__': [str(e)]}
                })
            raise
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class GroupCreateModalView(LoginRequiredMixin, CreateView):
    """
    View para criação de group via modal.
    """
    model = Group
    form_class = GroupForm
    template_name = 'task/modal_create_group.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Filtrar boards apenas do projeto atual se especificado
        board_id = self.request.GET.get('board')
        if board_id:
            try:
                from board.models import Board
                board = Board.objects.get(id=board_id)
                # Filtrar apenas o board atual
                context['form'].fields['board'].queryset = Board.objects.filter(
                    id=board.id
                )
            except Board.DoesNotExist:
                # Se o board não existir, mostrar todos os boards ativos
                context['form'].fields['board'].queryset = Board.objects.filter(
                    ativo=True
                ).order_by('project__nome', 'nome')
        else:
            # Se não há board especificado, mostrar todos os boards ativos
            context['form'].fields['board'].queryset = Board.objects.filter(
                ativo=True
            ).order_by('project__nome', 'nome')
        
        context.update({
            'modal_id': 'createGroupModal',
            'modal_title': 'Novo Grupo',
            'modal_icon': 'bi-collection',
            'form_id': 'groupForm',
            'submit_text': 'Criar Grupo'
        })
        return context
    
    def form_valid(self, form):
        group = form.save()
        
        # Registrar log
        log_object_created(self.request, group)
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Grupo criado com sucesso!',
                'obj': {
                    'id': group.id,
                    'nome': group.nome,
                    'board_id': group.board.id,
                    'board_name': group.board.nome
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)


class TaskUpdateModalView(LoginRequiredMixin, UpdateView):
    """
    View para edição de task via modal.
    """
    model = Task
    form_class = TaskForm
    template_name = 'task/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        task = self.get_object()
        
        # Garantir que o queryset dos grupos esteja definido
        context['form'].fields['group'].queryset = Group.objects.filter(
            ativo=True
        ).order_by('board__nome', 'nome')
        
        context.update({
            'modal_id': 'editTaskModal',
            'modal_title': f'Editar Tarefa: {task.nome}',
            'modal_icon': 'bi-list-task',
            'form_id': 'taskEditForm',
            'submit_text': 'Atualizar Tarefa'
        })
        return context
    
    def form_valid(self, form):
        task = form.save()
        
        # Registrar log
        log_object_updated(self.request, task)
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Tarefa atualizada com sucesso!',
                'obj': {
                    'id': task.id,
                    'nome': task.nome,
                    'status': task.get_status_display(),
                    'prioridade': task.get_prioridade_display(),
                    'group_name': task.group.nome if task.group else '',
                    'responsavel_name': task.responsavel.nome if task.responsavel else '',
                    'detail_url': reverse_lazy('task:detail', kwargs={'pk': task.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)
