#!/usr/bin/env python3
"""
Suite de testes para o sistema de gestão de agência.
Este script permite executar diferentes tipos de testes de forma organizada.
"""

import os
import sys
import subprocess
import argparse
from datetime import datetime

class TestSuite:
    """Classe para gerenciar diferentes tipos de testes"""
    
    def __init__(self):
        self.test_types = {
            'unit': 'Testes unitários',
            'integration': 'Testes de integração', 
            'api': 'Testes de API',
            'model': 'Testes de modelos',
            'view': 'Testes de views',
            'form': 'Testes de formulários',
            'all': 'Todos os testes'
        }
    
    def run_command(self, command, description=""):
        """Executa um comando e exibe o resultado"""
        print(f"\n{'='*60}")
        print(f"🧪 {description}")
        print(f"{'='*60}")
        print(f"Comando: {command}")
        print("-" * 60)
        
        try:
            result = subprocess.run(
                command,
                shell=True,
                text=True,
                timeout=300
            )
            
            if result.returncode == 0:
                print("✅ Sucesso!")
            else:
                print("❌ Falha!")
                print(f"Erro: {result.stderr}")
            
            return result.returncode == 0
            
        except subprocess.TimeoutExpired:
            print("❌ Timeout: comando demorou mais de 5 minutos")
            return False
        except Exception as e:
            print(f"❌ Erro: {e}")
            return False
    
    def run_unit_tests(self):
        """Executa testes unitários"""
        print("\n🔬 EXECUTANDO TESTES UNITÁRIOS")
        print("=" * 50)
        
        apps = ['task', 'client', 'collaborator', 'project', 'board', 'portfolio', 'workspace', 'notification']
        all_passed = True
        
        for app in apps:
            command = f'python manage.py test {app} --pattern="*ModelTest*" --verbosity=2'
            if not self.run_command(command, f"Testes de modelo - {app}"):
                all_passed = False
        
        return all_passed
    
    def run_integration_tests(self):
        """Executa testes de integração"""
        print("\n🔗 EXECUTANDO TESTES DE INTEGRAÇÃO")
        print("=" * 50)
        
        command = 'python manage.py test --pattern="*IntegrationTest*" --verbosity=2'
        return self.run_command(command, "Testes de integração")
    
    def run_api_tests(self):
        """Executa testes de API"""
        print("\n🌐 EXECUTANDO TESTES DE API")
        print("=" * 50)
        
        command = 'python manage.py test --pattern="*APITest*" --verbosity=2'
        return self.run_command(command, "Testes de API")
    
    def run_model_tests(self):
        """Executa testes de modelos"""
        print("\n📊 EXECUTANDO TESTES DE MODELOS")
        print("=" * 50)
        
        command = 'python manage.py test --pattern="*ModelTest*" --verbosity=2'
        return self.run_command(command, "Testes de modelos")
    
    def run_view_tests(self):
        """Executa testes de views"""
        print("\n👁️ EXECUTANDO TESTES DE VIEWS")
        print("=" * 50)
        
        command = 'python manage.py test --pattern="*ViewsTest*" --verbosity=2'
        return self.run_command(command, "Testes de views")
    
    def run_form_tests(self):
        """Executa testes de formulários"""
        print("\n📝 EXECUTANDO TESTES DE FORMULÁRIOS")
        print("=" * 50)
        
        command = 'python manage.py test --pattern="*FormTest*" --verbosity=2'
        return self.run_command(command, "Testes de formulários")
    
    def run_all_tests(self):
        """Executa todos os testes"""
        print("\n🚀 EXECUTANDO TODOS OS TESTES")
        print("=" * 50)
        
        command = 'python manage.py test --verbosity=2'
        return self.run_command(command, "Todos os testes")
    
    def run_coverage(self):
        """Executa testes com cobertura"""
        print("\n📊 EXECUTANDO TESTES COM COBERTURA")
        print("=" * 50)
        
        commands = [
            ('coverage run --source="." manage.py test --verbosity=1', "Executando testes com cobertura"),
            ('coverage report --show-missing', "Gerando relatório de cobertura"),
            ('coverage html --directory=htmlcov', "Gerando relatório HTML")
        ]
        
        all_passed = True
        for command, description in commands:
            if not self.run_command(command, description):
                all_passed = False
        
        return all_passed
    
    def run_system_checks(self):
        """Executa verificações do sistema"""
        print("\n🔧 EXECUTANDO VERIFICAÇÕES DO SISTEMA")
        print("=" * 50)
        
        commands = [
            ('python manage.py check --deploy', "Verificação de deploy"),
            ('python manage.py check --database default', "Verificação de banco de dados"),
            ('python manage.py check --tags security', "Verificação de segurança")
        ]
        
        all_passed = True
        for command, description in commands:
            if not self.run_command(command, description):
                all_passed = False
        
        return all_passed
    
    def run_performance_tests(self):
        """Executa testes de performance"""
        print("\n⚡ EXECUTANDO TESTES DE PERFORMANCE")
        print("=" * 50)
        
        # Verificar se Locust está instalado
        try:
            import locust
        except ImportError:
            print("⚠️  Locust não instalado. Instalando...")
            self.run_command('pip install locust', "Instalando Locust")
        
        command = 'python manage.py runserver --noreload & sleep 5 && locust -f performance_tests.py --headless --users 5 --spawn-rate 1 --run-time 30s'
        return self.run_command(command, "Testes de performance")
    
    def run_specific_app_tests(self, app_name):
        """Executa testes de um app específico"""
        print(f"\n🎯 EXECUTANDO TESTES DO APP: {app_name.upper()}")
        print("=" * 50)
        
        command = f'python manage.py test {app_name} --verbosity=2'
        return self.run_command(command, f"Testes do app {app_name}")
    
    def show_menu(self):
        """Exibe menu de opções"""
        print("\n🧪 SUITE DE TESTES - SISTEMA DE GESTÃO DE AGÊNCIA")
        print("=" * 60)
        print("Escolha o tipo de teste:")
        
        for i, (key, description) in enumerate(self.test_types.items(), 1):
            print(f"{i}. {description} ({key})")
        
        print(f"{len(self.test_types) + 1}. Testes com cobertura")
        print(f"{len(self.test_types) + 2}. Verificações do sistema")
        print(f"{len(self.test_types) + 3}. Testes de performance")
        print(f"{len(self.test_types) + 4}. App específico")
        print("0. Sair")
        
        return input("\nDigite sua escolha: ")
    
    def run(self):
        """Executa a suite de testes"""
        if len(sys.argv) > 1:
            # Modo linha de comando
            parser = argparse.ArgumentParser(description='Suite de testes para o sistema de gestão de agência')
            parser.add_argument('test_type', choices=list(self.test_types.keys()) + ['coverage', 'system', 'performance'], 
                              help='Tipo de teste a executar')
            parser.add_argument('--app', help='App específico para testar')
            
            args = parser.parse_args()
            
            if args.test_type == 'coverage':
                return self.run_coverage()
            elif args.test_type == 'system':
                return self.run_system_checks()
            elif args.test_type == 'performance':
                return self.run_performance_tests()
            elif args.app:
                return self.run_specific_app_tests(args.app)
            else:
                return getattr(self, f'run_{args.test_type}_tests')()
        else:
            # Modo interativo
            while True:
                choice = self.show_menu()
                
                if choice == '0':
                    print("👋 Saindo...")
                    break
                elif choice == str(len(self.test_types) + 1):
                    self.run_coverage()
                elif choice == str(len(self.test_types) + 2):
                    self.run_system_checks()
                elif choice == str(len(self.test_types) + 3):
                    self.run_performance_tests()
                elif choice == str(len(self.test_types) + 4):
                    app = input("Digite o nome do app: ")
                    self.run_specific_app_tests(app)
                elif choice.isdigit() and 1 <= int(choice) <= len(self.test_types):
                    test_type = list(self.test_types.keys())[int(choice) - 1]
                    getattr(self, f'run_{test_type}_tests')()
                else:
                    print("❌ Opção inválida!")
                
                input("\nPressione Enter para continuar...")


def main():
    """Função principal"""
    suite = TestSuite()
    success = suite.run()
    
    if success is not None:  # Modo linha de comando
        if success:
            print("\n🎉 Testes executados com sucesso!")
            sys.exit(0)
        else:
            print("\n💥 Alguns testes falharam.")
            sys.exit(1)


if __name__ == '__main__':
    main() 