from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter, OrderingFilter

from .models import Workspace
from .serializers import (
    WorkspaceSerializer, 
    WorkspaceDetailSerializer, 
    WorkspaceListSerializer
)


class WorkspaceViewSet(viewsets.ModelViewSet):
    """
    ViewSet para operações CRUD de workspaces.
    """
    
    queryset = Workspace.objects.all()
    permission_classes = [IsAuthenticated]
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_fields = ['ativo', 'cliente']
    search_fields = ['nome', 'descricao', 'cliente__nome']
    ordering_fields = ['nome', 'cliente__nome', 'created_at']
    ordering = ['cliente__nome', 'nome']

    def get_serializer_class(self):
        if self.action == 'retrieve':
            return WorkspaceDetailSerializer
        if self.action == 'list':
            return WorkspaceListSerializer
        return WorkspaceSerializer

    def get_queryset(self):
        queryset = Workspace.objects.select_related('cliente').all()
        
        # Filtro por status ativo
        ativo = self.request.query_params.get('ativo', None)
        if ativo is not None:
            queryset = queryset.filter(ativo=ativo.lower() == 'true')
            
        # Filtro por cliente
        cliente_id = self.request.query_params.get('cliente_id', None)
        if cliente_id:
            queryset = queryset.filter(cliente_id=cliente_id)
            
        return queryset

    @action(detail=True, methods=['post'])
    def toggle_status(self, request, pk=None):
        """
        Alterna o status ativo/inativo do workspace.
        """
        workspace = self.get_object()
        workspace.ativo = not workspace.ativo
        workspace.save()
        
        serializer = self.get_serializer(workspace)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def portfolios(self, request, pk=None):
        """
        Retorna os portfolios do workspace.
        """
        workspace = self.get_object()
        portfolios = workspace.portfolios.filter(ativo=True)
        
        from portfolio.serializers import PortfolioListSerializer
        serializer = PortfolioListSerializer(portfolios, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def projects(self, request, pk=None):
        """
        Retorna os projetos do workspace.
        """
        workspace = self.get_object()
        projects = workspace.projects.filter(ativo=True)
        
        from project.serializers import ProjectListSerializer
        serializer = ProjectListSerializer(projects, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def boards(self, request, pk=None):
        """
        Retorna os boards do workspace.
        """
        workspace = self.get_object()
        boards = workspace.boards.filter(ativo=True)
        
        from board.serializers import BoardListSerializer
        serializer = BoardListSerializer(boards, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['get'])
    def statistics(self, request, pk=None):
        """
        Retorna estatísticas do workspace.
        """
        workspace = self.get_object()
        
        stats = {
            'total_portfolios': workspace.total_portfolios,
            'total_projects': workspace.total_projects,
            'total_boards': workspace.total_boards,
            'portfolios_por_status': {},
            'projects_por_status': {},
            'tasks_por_status': {},
        }
        
        # Estatísticas de portfolios por status
        for portfolio in workspace.portfolios.all():
            status_key = portfolio.status
            if status_key not in stats['portfolios_por_status']:
                stats['portfolios_por_status'][status_key] = 0
            stats['portfolios_por_status'][status_key] += 1
        
        # Estatísticas de projetos por status
        for project in workspace.projects.all():
            status_key = project.status
            if status_key not in stats['projects_por_status']:
                stats['projects_por_status'][status_key] = 0
            stats['projects_por_status'][status_key] += 1
        
        # Estatísticas de tarefas por status
        from task.models import Task
        tasks = Task.objects.filter(group__board__workspace=workspace)
        for task in tasks:
            status_key = task.status
            if status_key not in stats['tasks_por_status']:
                stats['tasks_por_status'][status_key] = 0
            stats['tasks_por_status'][status_key] += 1
        
        return Response(stats)

