from django.db import models
from client.models import Client


class Workspace(models.Model):
    """
    Modelo para representar áreas de trabalho (workspaces).
    Um workspace é o nível mais alto da hierarquia organizacional.
    """
    nome = models.CharField(
        max_length=200,
        verbose_name="Nome",
        help_text="Nome do workspace"
    )
    
    cliente = models.ForeignKey(
        Client,
        on_delete=models.CASCADE,
        verbose_name="Cliente",
        help_text="Cliente proprietário do workspace",
        related_name="workspaces"
    )
    
    descricao = models.TextField(
        blank=True,
        null=True,
        verbose_name="Descrição",
        help_text="Descrição do workspace"
    )
    
    cor = models.CharField(
        max_length=7,
        default="#007bff",
        verbose_name="Cor",
        help_text="Cor do workspace (formato hexadecimal)"
    )
    
    ativo = models.BooleanField(
        default=True,
        verbose_name="Ativo",
        help_text="Indica se o workspace está ativo"
    )
    
    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name="Criado em"
    )
    
    updated_at = models.DateTimeField(
        auto_now=True,
        verbose_name="Atualizado em"
    )

    class Meta:
        verbose_name = "Workspace"
        verbose_name_plural = "Workspaces"
        ordering = ['cliente__nome', 'nome']
        unique_together = ['cliente', 'nome']

    def __str__(self):
        return f"{self.cliente.nome} - {self.nome}"

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('workspace:detail', kwargs={'pk': self.pk})

    @property
    def total_portfolios(self):
        return self.portfolios.count()

    @property
    def total_projects(self):
        return self.projects.count()

    @property
    def total_boards(self):
        return self.boards.count()

