from rest_framework import serializers
from .models import Workspace
from client.serializers import ClientListSerializer


class WorkspaceSerializer(serializers.ModelSerializer):
    """
    Serializer para o modelo Workspace.
    """
    
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    total_portfolios = serializers.ReadOnlyField()
    total_projects = serializers.ReadOnlyField()
    total_boards = serializers.ReadOnlyField()
    
    class Meta:
        model = Workspace
        fields = [
            'id', 'nome', 'cliente', 'cliente_nome', 'descricao', 'cor', 'ativo',
            'total_portfolios', 'total_projects', 'total_boards',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']


class WorkspaceDetailSerializer(serializers.ModelSerializer):
    """
    Serializer detalhado para o modelo Workspace.
    """
    
    cliente = ClientListSerializer(read_only=True)
    portfolios = serializers.SerializerMethodField()
    projects = serializers.SerializerMethodField()
    boards = serializers.SerializerMethodField()
    
    class Meta:
        model = Workspace
        fields = [
            'id', 'nome', 'cliente', 'descricao', 'cor', 'ativo',
            'portfolios', 'projects', 'boards',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']
    
    def get_portfolios(self, obj):
        from portfolio.serializers import PortfolioListSerializer
        portfolios = obj.portfolios.filter(ativo=True)[:5]  # Últimos 5
        return PortfolioListSerializer(portfolios, many=True).data
    
    def get_projects(self, obj):
        from project.serializers import ProjectListSerializer
        projects = obj.projects.filter(ativo=True)[:5]  # Últimos 5
        return ProjectListSerializer(projects, many=True).data
    
    def get_boards(self, obj):
        from board.serializers import BoardListSerializer
        boards = obj.boards.filter(ativo=True)[:5]  # Últimos 5
        return BoardListSerializer(boards, many=True).data


class WorkspaceListSerializer(serializers.ModelSerializer):
    """
    Serializer simplificado para listagem de workspaces.
    """
    
    cliente_nome = serializers.CharField(source='cliente.nome', read_only=True)
    total_portfolios = serializers.ReadOnlyField()
    total_projects = serializers.ReadOnlyField()
    
    class Meta:
        model = Workspace
        fields = [
            'id', 'nome', 'cliente_nome', 'cor', 'ativo',
            'total_portfolios', 'total_projects'
        ]

