from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator
from django.db.models import Q, Count
from django.http import JsonResponse
from django.views.generic import CreateView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from .models import Workspace
from .forms import WorkspaceForm
from client.models import Client
from gestao_agencia.decorators import superuser_required
from gestao_agencia.mixins import SuperuserRequiredMixin

@login_required
def workspace_list(request):
    """
    Lista todos os workspaces com filtros e paginação.
    """
    # Obter parâmetros de filtro
    search = request.GET.get('search', '')
    status = request.GET.get('status', '')
    cliente_id = request.GET.get('cliente', '')
    
    # Query base - filtra apenas workspaces acessíveis (permissões diretas)
    from permissions.models import VisibilityPermission
    workspaces = VisibilityPermission.get_accessible_objects(request.user, Workspace)
    workspaces = workspaces.select_related('cliente')
    
    # Aplicar filtros
    if search:
        workspaces = workspaces.filter(
            Q(nome__icontains=search) |
            Q(cliente__nome__icontains=search) |
            Q(descricao__icontains=search)
        )
    
    if status:
        if status == 'ativo':
            workspaces = workspaces.filter(ativo=True)
        elif status == 'inativo':
            workspaces = workspaces.filter(ativo=False)
    
    if cliente_id:
        workspaces = workspaces.filter(cliente_id=cliente_id)
    
    # Ordenar por nome
    workspaces = workspaces.order_by('nome')
    
    # Paginação
    paginator = Paginator(workspaces, 12)  # 12 workspaces por página
    page_number = request.GET.get('page')
    page_obj = paginator.get_page(page_number)
    
    # Estatísticas (apenas dos workspaces acessíveis)
    workspaces_ativos = workspaces.filter(ativo=True).count()
    total_clientes = Client.objects.count()
    total_projetos = 0  # Será implementado quando o modelo Project estiver pronto
    
    # Lista de clientes para o filtro
    clientes = Client.objects.all().order_by('nome')
    
    context = {
        'workspaces': page_obj,
        'page_obj': page_obj,
        'is_paginated': page_obj.has_other_pages(),
        'search': search,
        'status': status,
        'cliente_id': cliente_id,
        'workspaces_ativos': workspaces_ativos,
        'total_clientes': total_clientes,
        'total_projetos': total_projetos,
        'clientes': clientes,
    }
    
    return render(request, 'workspace/list.html', context)

@login_required
@superuser_required
def workspace_create(request):
    """
    Cria um novo workspace.
    """
    if request.method == 'POST':
        form = WorkspaceForm(request.POST)
        if form.is_valid():
            workspace = form.save()
            messages.success(request, f'Workspace "{workspace.nome}" criado com sucesso!')
            return redirect('workspace:detail', pk=workspace.pk)
    else:
        form = WorkspaceForm()
    
    context = {
        'form': form,
        'workspace': None,
    }
    
    return render(request, 'workspace/form.html', context)

@login_required
@superuser_required
def workspace_edit(request, pk):
    """
    Edita um workspace existente.
    """
    workspace = get_object_or_404(Workspace, pk=pk)
    
    if request.method == 'POST':
        form = WorkspaceForm(request.POST, instance=workspace)
        if form.is_valid():
            workspace = form.save()
            messages.success(request, f'Workspace "{workspace.nome}" atualizado com sucesso!')
            return redirect('workspace:detail', pk=workspace.pk)
    else:
        form = WorkspaceForm(instance=workspace)
    
    context = {
        'form': form,
        'workspace': workspace,
    }
    
    return render(request, 'workspace/form.html', context)

@login_required
def workspace_detail(request, pk):
    """
    Exibe os detalhes de um workspace.
    """
    workspace = get_object_or_404(Workspace, pk=pk)
    
    # Verifica se o usuário tem acesso ao workspace
    from permissions.models import VisibilityPermission
    can_access = VisibilityPermission.has_access(request.user, workspace)
    
    if not can_access:
        messages.error(request, "Você não tem permissão para acessar este workspace.")
        return redirect('workspace:list')
    
    # Verifica se o usuário pode gerenciar o workspace (apenas admin)
    can_manage = request.user.is_superuser
    
    # Obter portfolios e projetos do workspace
    portfolios = workspace.portfolios.filter(ativo=True).order_by('nome')[:5]  # Últimos 5
    projects = workspace.projects.filter(ativo=True).order_by('-created_at')[:5]  # Últimos 5
    
    context = {
        'workspace': workspace,
        'can_manage': can_manage,
        'can_access': can_access,
        'portfolios': portfolios,
        'projects': projects,
    }
    
    return render(request, 'workspace/detail.html', context)

@login_required
@superuser_required
def workspace_delete(request, pk):
    """
    Exclui um workspace.
    """
    workspace = get_object_or_404(Workspace, pk=pk)
    
    if request.method == 'POST':
        nome = workspace.nome
        workspace.delete()
        messages.success(request, f'Workspace "{nome}" excluído com sucesso!')
        return redirect('workspace:list')
    
    context = {
        'workspace': workspace,
    }
    
    return render(request, 'workspace/delete.html', context)


class WorkspaceCreateModalView(LoginRequiredMixin, SuperuserRequiredMixin, CreateView):
    """
    View para criação de workspace via modal.
    """
    model = Workspace
    form_class = WorkspaceForm
    template_name = 'workspace/modal_create.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'createWorkspaceModal',
            'modal_title': 'Novo Workspace',
            'modal_icon': 'bi-briefcase',
            'form_id': 'workspaceForm',
            'submit_text': 'Criar Workspace'
        })
        return context
    
    def form_valid(self, form):
        workspace = form.save()
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Workspace criado com sucesso!',
                'obj': {
                    'id': workspace.id,
                    'nome': workspace.nome,
                    'cliente': workspace.cliente.nome,
                    'detail_url': reverse_lazy('workspace:detail', kwargs={'pk': workspace.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)

class WorkspaceUpdateModalView(LoginRequiredMixin, SuperuserRequiredMixin, UpdateView):
    """
    View para edição de workspace via modal.
    """
    model = Workspace
    form_class = WorkspaceForm
    template_name = 'workspace/modal_edit.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context.update({
            'modal_id': 'editWorkspaceModal',
            'modal_title': 'Editar Workspace',
            'modal_icon': 'bi-pencil',
            'form_id': 'workspaceEditForm',
            'submit_text': 'Salvar Alterações'
        })
        return context
    
    def form_valid(self, form):
        workspace = form.save()
        
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': True,
                'message': 'Workspace atualizado com sucesso!',
                'obj': {
                    'id': workspace.id,
                    'nome': workspace.nome,
                    'cliente': workspace.cliente.nome,
                    'detail_url': reverse_lazy('workspace:detail', kwargs={'pk': workspace.id})
                }
            })
        return super().form_valid(form)
    
    def form_invalid(self, form):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            return JsonResponse({
                'success': False,
                'errors': form.errors
            })
        return super().form_invalid(form)
